/*
 * File: SeeingStars.java
 * =============================================================
 * A program that draws pretty stars on the screen!
 */
import acm.graphics.*;
import acm.program.*;
import java.awt.*;
import java.util.*;
import javax.swing.*;
import java.awt.event.*;

public class SeeingStars extends GraphicsProgram {
	/* The number of points in the star. */
	private static final int NUM_STAR_POINTS = 7;
	
	/* The radius of the star. */
	private static final double STAR_RADIUS = 150;

	/* A map associating names with Color objects representing those colors. */
	private HashMap<String, Color> colorMap;
	
	/* The color chooser Combo Box. */
	private JComboBox colorChooser;
	
	/* The checkbox controlling the background. */
	private JCheckBox backgroundChooser;
	
	/* Radio buttons to choose the number of points */
	private JRadioButton five, seven, nine;
	
	/* How many points are on the star. */
	private int numPoints = NUM_STAR_POINTS;
	
	public void init() {
		populateColorMap();
		
		/* Add the colors from the color map to the combo box. */
		colorChooser = new JComboBox();
		for (String key: colorMap.keySet()) {
			colorChooser.addItem(key);
		}
		
		/* Set the default option to "Blue." */
		colorChooser.setSelectedItem("Blue");
		
		/* Make it so that we can't pick a color that isn't there. */
		colorChooser.setEditable(false);
		colorChooser.addActionListener(this);
		add(colorChooser, SOUTH);
		
		/* Set up the background chooser. */
		backgroundChooser = new JCheckBox("Dark background");
		backgroundChooser.addActionListener(this);
		add(backgroundChooser, SOUTH);
		
		/* Configure the radio buttons for the number of points. */
		five = new JRadioButton("5");
		seven = new JRadioButton("7");
		nine = new JRadioButton("9");
		
		five.addActionListener(this);
		seven.addActionListener(this);
		nine.addActionListener(this);
		
		ButtonGroup group = new ButtonGroup();
		group.add(five);
		group.add(seven);
		group.add(nine);
		
		add(five, SOUTH);
		add(seven, SOUTH);
		add(nine, SOUTH);
	}
	
	/**
	 * Based on the command we get, update the star appropriately.
	 */
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == colorChooser) {
			redrawStar();
		} else if (e.getSource() == backgroundChooser) {
			setBackground(backgroundChooser.isSelected()? Color.BLACK : Color.WHITE);
		} else if (e.getSource() == five) {
			numPoints = 5;
			redrawStar();
		} else if (e.getSource() == seven) {
			numPoints = 7;
			redrawStar();
		} else if (e.getSource() == nine) {
			numPoints = 9;
			redrawStar();
		}
	}
	
	/**
	 * Updates the displayed star.
	 */
	private void redrawStar() {
		removeAll();
		
		/* From the combo box, obtain the selected color name.  Then, using the
		 * map of color names to Color objects, determine the color of the star.
		 */
		Color starColor = colorMap.get(colorChooser.getSelectedItem());
		drawStar(starColor, numPoints);
	}
	
	/**
	 * Draws a pretty star!
	 */
	public void run() {
		redrawStar();
	}
	
	/**
	 * Fills in the colorMap field with an assortment of colors.
	 */
	private void populateColorMap() {
		colorMap = new HashMap<String, Color>();
		colorMap.put("Black", Color.BLACK);
		colorMap.put("Green", Color.GREEN);
		colorMap.put("Blue", Color.BLUE);
		colorMap.put("Cyan", Color.CYAN);
		colorMap.put("Red", Color.RED);
		colorMap.put("Yellow", Color.YELLOW);
		colorMap.put("Magenta", Color.MAGENTA);
		colorMap.put("White", Color.WHITE);
	}
	
	/**
	 * Draws a star with the given color and number of points, centered
	 * horizontally in the screen,
	 * 
	 * @param c The color for the star.
	 * @param numPoints How many points the star has.
	 */
	private void drawStar(Color c, int numPoints) {
		double x = getWidth() / 2.0;
		double y = getHeight() / 2.0;
		
		for (int i = 0; i < numPoints; i++) {
			/* Compute the angles of the current point and the next point. */
			double theta     =  i      * 2 * Math.PI / numPoints;
			double nextTheta = (i + 2) * 2 * Math.PI / numPoints;
			
			/* Construct a line between those points. */
			GLine line = new GLine(x + STAR_RADIUS * Math.cos(theta),
					               y - STAR_RADIUS * Math.sin(theta),
					               x + STAR_RADIUS * Math.cos(nextTheta),
					               y - STAR_RADIUS * Math.sin(nextTheta));
			
			/* Update the line color. */
			line.setColor(c);
			
			/* Display the line. */
			add(line);			
		}
	}
}